<?php
namespace Hugeicons;

class Loader {
	private $admin;

	public function init() {
		if (is_admin()) {
			$this->admin = new Admin();
			
			// Add filter for plugin action links
			add_filter('plugin_action_links_hugeicons/hugeicons.php', array($this, 'add_plugin_action_links'));
		}

		add_action('wp_enqueue_scripts', array($this, 'enqueue_frontend_assets'));
		add_action('enqueue_block_editor_assets', array($this, 'enqueue_editor_assets'));
		add_shortcode('hugeicons', array($this, 'render_shortcode'));
	}

	public function enqueue_frontend_assets() {
		// Allow pro plugin to handle asset loading
		$should_load_free = apply_filters('hugeicons_enqueue_assets', true);
		
		if ($should_load_free) {
			wp_enqueue_style(
				'hugeicons-free',
				HUGEICONS_PLUGIN_URL . 'css/font/icons.css',
				array(),
				HUGEICONS_VERSION
			);
		}
	}

	public function render_shortcode($atts) {
		$atts = shortcode_atts(array(
			'icon' => '',
			'variant' => 'stroke',
			'type' => '',
			'size' => 'inherit',
			'color' => 'inherit',
		), $atts, 'hugeicons');

		if ($atts['type']) {
			$atts['type'] = '-' . esc_attr($atts['type']);
		}

		return sprintf(
			'<i class="hgi-%s%s hgi-%s" style="font-size: %s; color: %s"></i>',
			esc_attr($atts['variant']),
			esc_attr($atts['type']),
			esc_attr($atts['icon']),
			esc_attr($atts['size'] . 'px'),
			esc_attr($atts['color'])
		);
	}

	/**
	 * Enqueue block editor assets for the Hugeicons plugin.
	 */
	public function enqueue_editor_assets() {
		$format_api_file = HUGEICONS_PLUGIN_DIR . 'build/hugeicons-picker.asset.php';

		if (file_exists($format_api_file)) {
			$assets = include $format_api_file;
			wp_enqueue_script(
				'hugeicons-editor-js',
				HUGEICONS_PLUGIN_URL . 'build/hugeicons-picker.js',
				$assets['dependencies'],
				$assets['version']
			);

			wp_enqueue_style(
				'hugeicons-editor-css',
				HUGEICONS_PLUGIN_URL . 'build/style-hugeicons-picker.css',
				array(),
				HUGEICONS_VERSION
			);

			// Define available styles for free version
			$free_styles = array(
				'stroke-rounded',  // Basic stroke style
			);

			// Allow pro version to add more styles
			$available_styles = apply_filters('hugeicons_available_styles', $free_styles);

			// Localize the script with available styles
			wp_localize_script(
				'hugeicons-editor-js',
				'hugeiconStyles',
				$available_styles
			);
		}
	}

	/**
	 * Add plugin action links
	 * 
	 * @param array $links Existing plugin action links
	 * @return array Modified plugin action links
	 */
	public function add_plugin_action_links($links) {
		$settings_link = sprintf(
			'<a href="%s">%s</a>',
			admin_url('admin.php?page=hugeicons-settings'),
			__('Settings', 'hugeicons')
		);
		
		$links[] = $settings_link;

		// Add upgrade link if pro is not active
		if (!class_exists('HugeiconsPro\\Loader')) {
			$upgrade_link = sprintf(
				'<a href="%s" target="_blank" style="color: #10B981; font-weight: 600;">%s</a>',
				'https://hugeicons.com/pricing',
				__('Upgrade to Pro', 'hugeicons')
			);
			$links[] = $upgrade_link;
		}

		return $links;
	}
}