<?php
namespace Hugeicons;

class Admin {
	public function __construct() {
		add_action('admin_menu', array($this, 'add_plugin_page'));
		add_action('admin_init', array($this, 'register_settings'));
		add_action('admin_enqueue_scripts', array($this, 'enqueue_assets'));
	}

	public function add_plugin_page() {
		add_options_page(
			'Hugeicons Settings',
			'Hugeicons',
			'manage_options',
			'hugeicons-settings',
			array($this, 'render_admin_page')
		);
	}

	public function register_settings() {
		register_setting(
			'hugeicons_settings_group',
			'hugeicons_settings',
			array(
				'sanitize_callback' => array($this, 'sanitize_settings'),
				'default' => array()
			)
		);
	}

	/**
	 * Sanitize settings before saving to database
	 *
	 * @param array $input The settings input array
	 * @return array Sanitized settings
	 */
	public function sanitize_settings($input) {
		$sanitized = array();
		
		if (is_array($input)) {
			foreach ($input as $key => $value) {
				if (is_array($value)) {
					$sanitized[$key] = array_map('sanitize_text_field', $value);
				} else {
					$sanitized[$key] = sanitize_text_field($value);
				}
			}
		}
		
		return $sanitized;
	}

	public function enqueue_assets($hook) {
		if ('settings_page_hugeicons-settings' !== $hook) {
			return;
		}

		wp_enqueue_style(
			'hugeicons-admin',
			HUGEICONS_PLUGIN_URL . 'css/admin.css',
			array(),
			HUGEICONS_VERSION
		);

		wp_enqueue_script(
			'hugeicons-admin',
			HUGEICONS_PLUGIN_URL . 'js/admin.js',
			array('jquery'),
			HUGEICONS_VERSION,
			true
		);

		wp_localize_script('hugeicons-admin', 'hugeiconSettings', array(
			'nonce' => wp_create_nonce('hugeicons_nonce'),
			'ajaxUrl' => admin_url('admin-ajax.php')
		));
	}

	public function render_admin_page() {
		if (!current_user_can('manage_options')) {
			return;
		}

		// Get settings tabs from free and pro plugins
		$tabs = apply_filters('hugeicons_settings_tabs', array(
			'free' => array(
				'label' => __('Free Icons', 'hugeicons'),
				'callback' => array($this, 'render_free_settings')
			)
		));

		$active_tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'free';
		?>
		<div class="wrap hugeicons-settings">
			<h1><?php echo esc_html(get_admin_page_title()); ?></h1>

			<nav class="nav-tab-wrapper">
				<?php foreach ($tabs as $tab_id => $tab) : ?>
					<a href="?page=hugeicons-settings&tab=<?php echo esc_attr($tab_id); ?>" 
					   class="nav-tab <?php echo $active_tab === $tab_id ? 'nav-tab-active' : ''; ?>">
						<?php echo esc_html($tab['label']); ?>
					</a>
				<?php endforeach; ?>
			</nav>

			<div class="tab-content">
				<?php
				if (isset($tabs[$active_tab]['callback'])) {
					call_user_func($tabs[$active_tab]['callback']);
				}
				?>
			</div>
		</div>
		<?php
	}

	public function render_free_settings() {
		?>
		<div class="hugeicons-card">
			<h2><?php esc_html_e('Free Icons', 'hugeicons'); ?></h2>
			<p><?php esc_html_e('You are currently using the free version of Hugeicons. You have access to our free icon set.', 'hugeicons'); ?></p>
			
			<?php if (!class_exists('HugeiconsPro\\Loader')) : ?>
				<div class="hugeicons-upgrade-notice">
					<h3><?php esc_html_e('Upgrade to Pro', 'hugeicons'); ?></h3>
					<p><?php esc_html_e('Get access to our premium icon sets and advanced features:', 'hugeicons'); ?></p>
					<ul>
						<li><?php esc_html_e('Access to all premium icon sets', 'hugeicons'); ?></li>
						<li><?php esc_html_e('Custom icon set selection', 'hugeicons'); ?></li>
						<li><?php esc_html_e('Priority support', 'hugeicons'); ?></li>
					</ul>
					<a href="<?php echo esc_url('https://hugeicons.com/pricing'); ?>" target="_blank" class="button button-primary">
						<?php esc_html_e('Upgrade Now', 'hugeicons'); ?>
					</a>
				</div>
			<?php endif; ?>
		</div>
		<?php
	}
}